/** @file   war.cpp
 * @brief   Implementation of War - class.
 * @version $Revision: 1.17 $
 * @author  Tomi Lamminsaari
 */

#include <almp3.h>
#include "war.h"
#include "www_map.h"
#include "GfxManager.h"
#include "GfxId.h"
#include "consts.h"
#include "corpse.h"
#include "animplayer.h"
#include "dynamicevent.h"
#include "settings.h"
#include "weapon.h"
#include "warglobals.h"
#include "Gradient.h"
#include "www_assert.h"
#include "lighttable.h"
#include "ScoresAnimation.h"
#include "carnivorealien.h"
#include "smallwormalien.h"
#include "wingedalien.h"
#include "proctoralien.h"
#include "minigunalien.h"
#include "tank.h"
#include "car.h"
#include "tileid.h"
#include "tprinter.h"
#include "sound.h"
#include "soundsamples.h"
#include "level.h"
#include "basecontroller.h"
#include "electricfence.h"
#include "storyviewer.h"
#include "playercar.h"
#include "Resource/HudSymbols.rh"
#include <fstream>
#include "LightSourceModel.h"
#include "LightModelUids.h"
#include "ComboMeter.h"
#include "AnimId.h"
#include "MusicControlObject.h"
#include "CreditsViewer.h"
#include "LightBeam.h"
using namespace eng2d;
using std::vector;
using std::string;

namespace WeWantWar {

const int War::PLAYER_INDEX;

const int War::QUIT_PLAYER_DIED;
const int War::QUIT_ESC;
const int War::QUIT_LEVEL_COMPLETE;
const int War::QUIT_OBJECTIVE_FAILED;

const int War::SECTOR_COLUMNS;
const int War::SECTOR_ROWS;
const int KPlaybackVolume = 128;

War::War() :
  m_quitNow( 0 ),
  m_pWeather( 0 ),
  m_menupage( PAGE_GAME ),
  m_pObjectives( 0 ),
  m_pRedrawQueue( 0 ),
  m_currentObjective( 0 ),
  m_sectorVector( SECTOR_ROWS * SECTOR_COLUMNS ),
  m_pauseMenuDelay( 10 ),
  m_lightTable( 0 ),
  m_currentScanLine( 240 ),
  m_gameOverCounter( 0 )
{
  for ( int i=0; i < m_sectorVector.size(); i++ ) {
    m_sectorVector.at(i).reserve( 120 );
  }
}


War::~War()
{
}


int War::prepareLevel(int levelNum)
{
  m_levelNumber = levelNum;
  
  // We initialize the globals. This creates the ParticleManager,
  // LightFenceManager, Hud and ObjectStorage objects. This also
  // initializes the bullet time and resets the level statistics.
  WarGlobals::initGlobals();
  m_pObjManager = WarGlobals::pObjTable;
  m_pHud = WarGlobals::pHud;
 
 // Initialize the lights
  m_lightTable = new LightTable;
  this->createLightModels(); 
  
  // Load the level and the map. We store the level information to the
  // static WarGlobals - class so that other objects can easily get
  // access to it.
  string levelfile = Settings::levelList.getFile( levelNum );
  m_pLevel = new Level();
  if ( m_pLevel->load( levelfile ) != 0 ) {
    return -1;
  }
  WarGlobals::pLevel = m_pLevel;
  WarGlobals::timeBonuses = m_pLevel->timebonus;
  
  // Create the gameobjects
  this->createObjects();
  
  // Init the animation player
  AnimPlayer::init();
    
  // Init background music
  for ( int i=0; i < m_pLevel->musicfile.size(); i++ ) {
    m_playlist.push_back( new MP3File( m_pLevel->musicfile.at(i) ) );
  }
  m_playlist.at(0)->open();
  
  srand(time(0));
  
  // Init the weather effects.
  m_pWeather = 0;
  if ( m_pLevel->weather != 0 ) {
    m_pWeather = new Weather( m_pLevel->weather );
  }
  
  // Make sure the game is not in paused state at the beginning.
  m_paused = false;
  
  // Create the redraw queue
  m_pRedrawQueue = new RedrawQueue();
  
  // Load the pause-menu background
  m_pPauseBitmap = load_bitmap( "gfx/paused.bmp", 0 );
  m_pDeadTitle = load_bitmap( "gfx/deadtitle.bmp", 0 );
  m_pLevelCompleteTitle = load_bitmap( "gfx/completetitle.bmp", 0 );
  
  // Set the default target object
  m_pObjManager->pPrimaryTarget = m_pObjManager->pPlayer;
    
  // Initialize the Dynamic Shadows
  WarGlobals::pShadowManager->buildShadows();
  
  m_comboMeter = new ComboMeter( WarGlobals::gameStats );
  
  return 0;
}


void War::play()
{
  // Load the player's data from previus level
  if ( exists( Consts::TMPFILE_PLAYERDATA.c_str() ) ) {
    FileHasher hfile( Consts::TMPFILE_PLAYERDATA );
    if ( hfile.check() ) {
      alert("Invalid playerdata!","Using the defaults",0, "ok",0, 0,0);
    } else {
      std::ifstream fin( Consts::TMPFILE_PLAYERDATA.c_str() );
      m_pObjManager->pPlayer->loadPlayerData( fin );
      fin.close();
    }
  }
  
  // Show the story
  this->showLevelStory();
  
  if ( Settings::cheatAmmo == true ) {
    grantCheatBonuses();
  }
  
  // This is the main loop of the game. At first we update the game as many
  // times it's necassary to reach the 40 updates/second. Then we draw
  // the scenary and decode some part of the mp3-file.
  m_quitNow = 0;
  
  // First update is done before we start the mp3-playback. Because most of the
  // enemies are created when we first time call the update(), it may
  // take quite long especially if we have to load some graphics.
  this->update();
  this->update();
  m_menupage = PAGE_LEVELFADEIN;
  
  // If the background musics are turned on, start the playback
  if ( Settings::musicOn ) {
    MP3Player::setMP3( m_playlist.at(0) );
    MP3Player::play();
    MP3Player::volume( KPlaybackVolume );
  }
  Sound::minimumReplayDelay = 2;
  
  // Init the variables that control the game speed
  int frameSkip = 0;
  bool needsRedraw = false;
  
  //timerCounter = 0;         // Found in "consts.h"
  FpsTimer* pFps = FpsTimer::getInstance();
  
  LOG_MESSAGE( "STARTING GAME!" );
  while ( m_quitNow == 0 ) {
    frameSkip = 0;
    //if ( timerCounter > 0 ) {
    if ( pFps->updatesToGo() > 0 ) {
      do {
        // We update the game
        this->update();
        // Handle the constant game speed
        //timerCounter--;
        pFps->updateDone();
        frameSkip++;
        if (frameSkip >= 6) {
          pFps->reset();
          //timerCounter = 0;
          break;
        }
      
      } while ( pFps->updatesToGo() > 0 );
      needsRedraw = true;
    }
    
    if ( Settings::musicOn ) {
      if ( MP3Player::poll() == ALMP3_POLL_PLAYJUSTFINISHED ) {
        MP3Player::play();
      }
    }
        
    // Update the screen if necessary
    rest(0);
    if ( needsRedraw ) {
      needsRedraw = false;
      this->redraw();
      Display::flip();
    }
    
    if ( key[KEY_F5] ) {
      showDebugInfo();
    }
    
    // We have some special keys that operate only, when we're running in
    // debugging mode.
#ifndef NDEBUG
    if ( key[KEY_F10] ) {
      this->whileKey( KEY_F10 );
      WarGlobals::freeScrollMode = !WarGlobals::freeScrollMode;
    }  
#endif

  } // end of mainloop
  
  FpsTimer::releaseInstance( pFps );
  
  // Stop the sound playback
  Sound::defaultFrequency = 1000;
  Sound::stopAllSamples();
  Sound::clearSoundSources();
  Sound::minimumReplayDelay = -1;
  if ( Settings::musicOn ) {
    MP3Player::stop();
    MP3Player::setMP3( 0 );
  }
  
  // Show the levelcomplete menu if level was completed
  if ( m_quitNow == QUIT_LEVEL_COMPLETE ) {
    // If this was the last level, show credits
    if ( m_levelNumber == Settings::levelList.filecount() - 1 ) {
      showCredits();
    }
    
    // Grant time bonus
    if ( WarGlobals::timeBonuses > 0 ) {
      WarGlobals::gameStats.grantScores( WarGlobals::timeBonuses );
    }
    m_menupage = PAGE_LEVELCOMPLETE;
    Sound::playSample( SMP_MENUSELECT, false );
    while ( !( key[KEY_ENTER] || key[KEY_SPACE] ) ) {
      this->redraw();
      Display::flip();
      int r = retrace_count + 1;
      while ( retrace_count < r );
    }
    Sound::playSample( SMP_MENUSELECT, false );
    while ( key[KEY_ENTER] || key[KEY_SPACE] );
  }
  
  // Save the playerdata
  if ( m_quitNow == QUIT_LEVEL_COMPLETE ) {
    std::ofstream fout( Consts::TMPFILE_PLAYERDATA.c_str() );
    m_pObjManager->pPlayer->savePlayerData( fout );
    
    fout << "[END]" << std::endl;
    fout.close();
  
    FileHasher hfile( Consts::TMPFILE_PLAYERDATA );
    hfile.set();
  }
}


void War::destroyLevel()
{
  delete m_pLevel;
  m_pLevel = 0;
  
  // Now we cleanup the objects created by the WarGlobals - class during
  // the level initialization.
  WarGlobals::cleanupGlobals();
  
  
  // Destroys bullets, grenades, animations and lights
  AnimPlayer::cleanup();
  delete m_lightTable;  m_lightTable = 0;

  // Destroy the playlist
  for (int i=0; i < m_playlist.size(); i++) {
    delete m_playlist.at(i);
    m_playlist.at(i) = 0;
  }
  m_playlist.clear();


  // Destroy the weather effects
  delete m_pWeather;
  m_pWeather = 0;

  // Destroy the level objective-notes.
  delete m_pObjectives;
  m_pObjectives = 0;

  // Destroy the redraw queue
  delete m_pRedrawQueue;
  m_pRedrawQueue = 0;

  delete m_comboMeter;
  
  // Remove the pause-menu background image
  if ( m_pPauseBitmap != 0 ) {
    destroy_bitmap( m_pPauseBitmap );
    m_pPauseBitmap = 0;
  }
  if ( m_pDeadTitle != 0 ) {
    destroy_bitmap( m_pDeadTitle );
    m_pDeadTitle = 0;
  }
  if ( m_pLevelCompleteTitle != 0 ) {
    destroy_bitmap( m_pLevelCompleteTitle );
    m_pLevelCompleteTitle = 0;
  }
}

int War::getEndReason() const
{
  return m_quitNow;
}


void War::createObjects()
{
  // First we create the player.
  Player* pPlayer = new Player();
  Vec2D pos;
  pos.vx = m_pLevel->playerStartingPos.plaX * Map::getBlockWidth();
  pos.vy = m_pLevel->playerStartingPos.plaY * Map::getBlockHeight();
  
  pPlayer->position( pos );
  pPlayer->angle( m_pLevel->playerStartingPos.plaAngle );
  pPlayer->objectID( m_pLevel->playerStartingPos.plaID );
  
  m_pObjManager->addObject( pPlayer );
  
  // Create the music playback controller object
  MusicControlObject* controlObj = new MusicControlObject( *this );
  controlObj->objectID( KMp3PlayerControlObjectId );
  m_pObjManager->addObject( controlObj );
  
  // Create the aliens
  this->createAlienObjects();
}


void War::createAlienObjects()
{
  Map::selectLayer( Map::EObjects );
  int index = 0;
  int pixX, pixY;
  for (int mx=0; mx < Map::getWidth(Map::IN_BLOCKS); mx++) {
    pixX = mx * Map::getBlockWidth();
    
    for (int my=0; my < Map::getHeight(Map::IN_BLOCKS); my++) {
      pixY = my * Map::getBlockHeight();
      
      GameObject* pObj = 0;
      BonusObject* pBonus = 0;
      
      Map::Block* pB = Map::blockAt(mx,my, Map::IN_BLOCKS);
      int id = pB->user5;

      if ( id == TileID::id[TileID::ID_ALIEN1] ) {
        pObj = new CarnivoreAlien();

      } else if ( id == TileID::id[TileID::ID_ALIEN2] ) {
        pObj = new SmallWormAlien();

      } else if ( id == TileID::id[TileID::ID_ALIEN3] ) {
        pObj = new WingedAlien();
          
      } else if ( id == TileID::id[TileID::ID_ALIEN4] ) {
        pObj = 0;

      } else if ( id == TileID::id[TileID::ID_VS_DOOR] ||
                  id == TileID::id[TileID::ID_HS_DOOR] ) {
        pObj = 0;
        Door* pD = new Door( Door::SMALL, mx,my );
        WarGlobals::doorList.push_back( pD );

      } else if ( id == TileID::id[TileID::ID_BONUS_RIFLE] ) {
        pObj = 0;
        pBonus = new BonusObject( BonusObject::RIFLE );
        pBonus->size(22);

      } else if ( id == TileID::id[TileID::ID_BONUS_SHOTGUN] ) {
        pObj = 0;
        pBonus = new BonusObject( BonusObject::SHOTGUN );
        pBonus->size(22);

      } else if ( id == TileID::id[TileID::ID_BONUS_FIRSTAID] ) {
        pObj = 0;
        pBonus = new BonusObject( BonusObject::FIRSTAID );
        pBonus->size(22);

      } else if ( id == TileID::id[TileID::ID_BONUS_GRENADE] ) {
        pObj = 0;
        pBonus = new BonusObject( BonusObject::GRENADE );
        pBonus->size(22);

      } else if ( id == TileID::id[TileID::ID_BONUS_FLAMETHROWER] ) {
        pObj = 0;
        pBonus = new BonusObject( BonusObject::FLAMETHROWER );
        pBonus->size(22);

      } else if ( id == TileID::id[TileID::ID_ALIEN5] ) {
        pObj = new ProctorAlien();

      } else if ( id == TileID::id[TileID::ID_BONUS_MINIGUN] ) {
        pObj = 0;
        pBonus = new BonusObject( BonusObject::MINIGUN );
        pBonus->size( 22 );

      } else if ( id == TileID::id[TileID::ID_ALIEN6] ) {
        pObj = new MinigunAlien();

      } else if ( id == TileID::id[TileID::ID_BONUS_UZI] ) {
        pObj = 0;
        pBonus = new BonusObject( BonusObject::UZI );
        pBonus->size( 18 );

      } else if ( id == TileID::id[TileID::ID_STATIC_LIGHT] ) {
        pObj = 0;
        Vec2D pos( pixX + 16, pixY + 16 );
        LightSource* light = new LightSource( pos, LightModelUid::KStreetLight,
                                              128 );
        m_lightTable->add( light );

      } else {
        pObj = 0;
        pBonus = 0;
        
      }

      // Add the object to the objectlist.
      if ( pObj != 0 ) {
        pObj->position( Vec2D(pixX,pixY) );
        pObj->angle( rand() % 255 );
        Map::selectLayer( Map::EBackground );
        translateToFreeBlocks( pObj );
        Map::selectLayer( Map::EObjects );
        m_pObjManager->addObject( pObj );
      }
      if ( pBonus != 0 ) {
        pBonus->position( Vec2D(pixX, pixY) );
        WarGlobals::bonusList.push_back( pBonus );
      }
      index++;
    }
  }
  Map::selectLayer( Map::EBackground );
}


void War::translateToFreeBlocks( GameObject* pO )
{
  Vec2D p = pO->position();
  if ( pO->hasMapCollisionAt( p ) == false ) {
    return;
  }
  
  Vec2D newPos = p + Vec2D( 20,0 );
  if ( pO->hasMapCollisionAt( newPos ) == false ) {
    pO->position( newPos );
    return;
  }
  
  newPos = p + Vec2D( -20,0 );
  if ( pO->hasMapCollisionAt( newPos ) == false ) {
    pO->position( newPos );
    return;
  }
  
  newPos = p + Vec2D( 0,20 );
  if ( pO->hasMapCollisionAt( newPos ) == false ) {
    pO->position( newPos );
    return;
  }
  
  newPos = p + Vec2D( 0,-20 );
  if ( pO->hasMapCollisionAt( newPos ) == false ) {
    pO->position( newPos );
    return;
  }
}


void War::update()
{
  switch ( m_menupage ) {
    case ( PAGE_GAME ): {
      // Check if player wants to go to the pause menu
      m_pauseMenuDelay -= 1;
      if ( key[KEY_ESC] && m_pauseMenuDelay < 0 ) {
        whileKey( KEY_ESC );
        m_paused = true;
        m_menupage = PAGE_PAUSEMENU;
        m_selectedMenuItem = 0;
        m_lastKey = KEY_ESC;
        Sound::playSample( SMP_MENUSELECT, false );
        return;
      }
      if ( mouse_b & 2 ) {
        if ( m_bulletTimeButton == false && WarGlobals::bulletTimeLeft > 0 ) {
          MusicControlObject* musObj = dynamic_cast<MusicControlObject*>(
                m_pObjManager->findObject( KMp3PlayerControlObjectId ) );
          if ( WarGlobals::bulletTimeMode == false ) {
            WarGlobals::bulletTimeMode = true;
            Sound::playSample( SMP_BULLETTIME_BEGIN, false );
            Sound::playSample( SMP_HEARTBEAT, true );
            if ( musObj != 0 ) {
              musObj->launchVolumeFade(KPlaybackVolume/2, 10);
            }
//            MP3Player::volume( 64 );
            Sound::defaultFrequency = 820;
          } else {
            WarGlobals::bulletTimeMode = false;
            Sound::defaultFrequency = 1000;
            Sound::playSample( SMP_BULLETTIME_END, false );
            Sound::stopSample( SMP_HEARTBEAT );
            if ( musObj != 0) {
              musObj->launchVolumeFade(KPlaybackVolume,10);
            }
//            MP3Player::volume( 128 );
          }
        }
        m_bulletTimeButton = true;
        
      } else {
        m_bulletTimeButton = false;
        
      }
      
      // Update the game.
      this->updateGame();
      break;
    }
    case ( PAGE_PAUSEMENU ): {
      this->updatePauseMenu();
      break;
    }
    case ( PAGE_PLAYERDIED ): {
      this->updatePauseMenu();
      break;
    }
    case ( PAGE_LEVELFADEIN ): {
      this->updateLevelFadeIn();
      break;
    }
    case ( PAGE_GAMEOVERSCREEN ): {
      this->updateGameOverScreen();
      break;
    }
    default: {
      alert( "Unexpected menupage failure!", 0,0, "Ok",0, 0,0 );
      abort();
      break;
    }
  }
}


void War::updateGame()
{
  // If we're in free scroll mode, we check the arrowkeys and scroll the screen.
  if ( WarGlobals::freeScrollMode == true ) {
    if ( key[ KEY_UP ] ) {
      Map::scrollY -= 3;
    } else if ( key[ KEY_DOWN ] ) {
      Map::scrollY += 3;
    }
    
    if ( key[ KEY_LEFT ] ) {
      Map::scrollX -= 3;
    } else if ( key[ KEY_RIGHT ] ) {
      Map::scrollX += 3;
    }
    
    if ( Map::scrollX < 0 ) {
      Map::scrollX = 0;
    }
    if ( Map::scrollY < 0 ) {
      Map::scrollY = 0;
    }
    if ( Map::scrollX - Display::scrWidth() >= Map::getWidth( Map::IN_PIXELS ) ) {
      Map::scrollX = Map::getWidth( Map::IN_PIXELS ) - Display::scrWidth() - 1;
    }
    if ( Map::scrollY - Display::scrHeight() >= Map::getHeight( Map::IN_PIXELS ) ) {
      Map::scrollY = Map::getHeight( Map::IN_PIXELS ) - Display::scrHeight() - 1;
    }
  }
  
  // Manage the bullet time.
  if ( WarGlobals::bulletTimeMode == true ) {
    FpsTimer* timerHandler = FpsTimer::getInstance();
    unsigned int numOfUpdates = timerHandler->logicalUpdatesCount();
    FpsTimer::releaseInstance( timerHandler );
    
    WarGlobals::bulletTimeLeft -= 1;
    if ( WarGlobals::bulletTimeLeft < 0 ) {
      // Out of bullet time. We turn the bullettime mode off
      WarGlobals::bulletTimeMode = false;
      Sound::defaultFrequency = 1000;
      Sound::playSample( SMP_BULLETTIME_END, false );
      Sound::stopSample( SMP_HEARTBEAT );
      MusicControlObject* musObj = dynamic_cast<MusicControlObject*>(
                m_pObjManager->findObject( KMp3PlayerControlObjectId ) );
      if (musObj != 0) {
        musObj->launchVolumeFade(KPlaybackVolume,10);
      }
    }
    if ( numOfUpdates % 2 == 0 ) {
      // When the bullet time is on we don't update anything else but
      // the player.
      m_pObjManager->pPlayer->update();
      return;
    }
  }
  
  WarGlobals::gameStats.updateLevelTimer();
  WarGlobals::timeBonuses -= 1;
  WarGlobals::pPartManager->update();
  
  // If level has been accomplished, we don't update.
  if ( m_quitNow == QUIT_LEVEL_COMPLETE ) {
    return;
  }

  // Update the map animations.
  Map::update();
  
  // Update the current objective and dynamic events.
  Objective* pObjective = m_pLevel->getObjective( m_currentObjective );
  Objective::State objStat = pObjective->update();

  if ( objStat == Objective::ACCOMPLISHED ) {
    m_currentObjective += 1;
    if ( m_currentObjective >= m_pLevel->objectiveCount() ) {
      // All the objective has been accomplished. We are ready to exit
      // this level.
      m_quitNow = QUIT_LEVEL_COMPLETE;
      m_currentObjective -= 1;
    }
    
  } else if ( objStat == Objective::FAILED ) {
    // This objective failed. Game over
    m_quitNow = QUIT_OBJECTIVE_FAILED;
    
  }
  
  // Update the gameobjects and add them to the correct sector vectors. We also
  // do the teleportations here.
  for ( int i=0; i < m_sectorVector.size(); i++ ) {
    m_sectorVector.at(i).clear();
  }
  m_pObjManager->updateObjects();
  
  for ( int i=0; i < m_pObjManager->objectList.size(); i++ ) {
    GameObject* pO = m_pObjManager->objectList.at(i);
    
    // Check the teleporters.
    for ( int k=0; k < WarGlobals::teleporterList.size(); k++ ) {
      WarGlobals::teleporterList.at(k)->teleport( pO );
    }
    
    Vec2D p = pO->position();
    int secNum = Utils::sectorNumber( p ) & 63;
    m_sectorVector.at(secNum).push_back( pO );
  }  
  
  // If player is dead, we set the flag.
  if ( m_pObjManager->pPlayer->state() == GameObject::STATE_KILLED ) {
    // Player died. We decrease the lifecount and if it reaches the zero,
    // the game is over.
    if ( Settings::cheatLifes == false ) {
      WarGlobals::numberOfLives -= 1;
    }
    if ( WarGlobals::numberOfLives > 0 ) {
      m_menupage = PAGE_PLAYERDIED;
      m_paused = true;
      m_selectedMenuItem = 0;
      m_lastKey = KEY_ESC;
    } else {
//      m_quitNow = QUIT_PLAYER_DIED;
      m_menupage = PAGE_GAMEOVERSCREEN;
      return;
    }
  }

  // Check the collisions between the player and aliens
  this->checkGameObjectCollisions();

  // Check the collisions between the bullets and aliens
  this->checkBulletCollisions();
  
  // Update the bullet tables
  WarGlobals::pBulletManager->update();
  
  // Update the animations
  AnimPlayer::update();
  
  // Update the positional sounds
  Sound::update( Vec2D( Map::scrollX+320, Map::scrollY+240 ) );
  
  // Update the doors as well. The doors are opened by the GameObject-class.
  for (int i=0; i < WarGlobals::doorList.size(); i++) {
    WarGlobals::doorList.at(i)->update();
  }
  
  // Check the bonuses
  Vec2D plaPos = m_pObjManager->pPlayer->position();
  for (int i=0; i < WarGlobals::bonusList.size(); i++) {
    BonusObject* b = WarGlobals::bonusList.at(i);
    if ( b != 0 ) {
      Vec2D distV = b->position() - plaPos;
      if ( distV.length() < b->size() ) {
        if ( m_pObjManager->pPlayer->pickupBonus( b ) ) {
          // Player picked the bonus up so we delete it.
          m_pHud->addMessage( Settings::getBonusObjectName( b->type() ), 90 );
          delete b;
          WarGlobals::bonusList.at(i) = 0;
        }
      }
    }
  }
  
  // Update the weather effects
  if ( m_pWeather != 0 ) {
    m_pWeather->update();
  }

  // Update the objectives notes
  if ( m_pObjectives != 0 ) {
    m_pObjectives->update();
  }

  // Is player willing to capture/exit a vehicle.
  if ( WarGlobals::captureVehicle && WarGlobals::captureFeatureOn ) {
    // Has player already captured a vehicle
    if ( m_pObjManager->pPlayer->getVehicle() != 0 ) {
      // Yes he has. Exit the vehicle
      MCapturable* pT = dynamic_cast<MCapturable*>( m_pObjManager->pPlayer->getVehicle() );
      if ( pT != 0 ) {
        pT->capture( 0 );
      }
      
    } else {
      this->findVehicleToCapture();
      
    }
    WarGlobals::captureVehicle = false;
  }
  
  // Now we prepare the HUD
  this->updateHud();
  
  // Update the light fences
  WarGlobals::pFenceManager->update();
  this->checkPlayerLightfenceCollisions();
  
  // Update the combometer
  ComboMeter::ComboLevel comboLevel = m_comboMeter->update();
  if ( comboLevel != ComboMeter::ENoCombo ) {
    int animUid = -1;
    int soundId = -1;
    switch (comboLevel) {
      case ( ComboMeter::ETripleKill ): {
        animUid = 0;
        soundId = SMP_COMBO_TRIPLEKILL;
        break;
      }
      case ( ComboMeter::EButcher ): {
        animUid = 1;
        soundId = SMP_COMBO_BUTCHER;
        break;
      }
      case ( ComboMeter::ESerialKiller ): {
        animUid = 2;
        soundId = SMP_COMBO_SERIALKILLER;
        break;
      }
      case ( ComboMeter::EAlienMassacre ): {
        animUid = 3;
        soundId = SMP_COMBO_ALIENMASSACRE;
        break;
      }
      default: {
        animUid = -1; 
        break;
      }
    }
    if ( animUid != -1 ) {
      const Animation& anim = GameAnims::findAnimation( AnimId::KComboLabels, animUid );
      Animation* newAnim = new Animation;
      (*newAnim) = anim;
      m_pHud->setComboAnimation( newAnim );
    }
    if ( soundId != -1 ) {
      Sound::playSample( soundId, false );
    }
  }
}


void War::updatePauseMenu()
{
  if ( key[m_lastKey] ) {
    return;
  }
  m_lastKey = KEY_P;
  
  if ( key[KEY_UP] ) {
    m_selectedMenuItem -= 1;
    if ( m_selectedMenuItem < 0 ) {
      m_selectedMenuItem = 2;
    }
    m_lastKey = KEY_UP;
    Sound::playSample( SMP_MENUITEM, false );
    
  } else if ( key[KEY_DOWN] ) {
    m_selectedMenuItem += 1;
    if ( m_selectedMenuItem > 2 ) {
      m_selectedMenuItem = 0;
    }
    m_lastKey = KEY_DOWN;
    Sound::playSample( SMP_MENUITEM, false );
    
  } else if ( key[KEY_SPACE] ) {
    m_lastKey = KEY_SPACE;
    if ( m_selectedMenuItem == 0 ) {
      // Resume / resurrect the game
      if ( m_menupage == PAGE_PLAYERDIED ) {
        // We wakeup the player.
        m_pObjManager->pPlayer->resurrect();
        WarGlobals::bulletTimeLeft = Consts::MAX_BULLETTIME;
      }
      m_paused = false;
      m_menupage = PAGE_GAME;
      
    } else if ( m_selectedMenuItem == 2 ) {
      m_paused = false;
      m_menupage = PAGE_GAME;
      m_quitNow = QUIT_ESC;
    }
    Sound::playSample( SMP_MENUSELECT, false );
    
  } else if ( key[KEY_ENTER] ) {
    m_lastKey = KEY_ENTER;
    if ( m_selectedMenuItem == 0 ) {
      // Resume / resurrect the game
      if ( m_menupage == PAGE_PLAYERDIED ) {
        // We wakeup the player.
        m_pObjManager->pPlayer->resurrect();
        WarGlobals::bulletTimeLeft = Consts::MAX_BULLETTIME;
      }
      m_paused = false;
      m_menupage = PAGE_GAME;
    } else if ( m_selectedMenuItem == 2 ) {
      m_paused = false;
      m_menupage = PAGE_GAME;
      m_quitNow = QUIT_ESC;
    }
    Sound::playSample( SMP_MENUSELECT, false );
  }
}


void War::updateLevelFadeIn()
{
  m_currentScanLine -= 5;
  if ( m_currentScanLine < 1 ) {
    m_menupage = PAGE_GAME;
  }
}

void War::updateGameOverScreen()
{
  m_gameOverCounter += 1;
  if ( m_gameOverCounter > 180 ) {
    m_quitNow = QUIT_PLAYER_DIED;
  }
}

void War::checkGameObjectCollisions()
{
  if ( m_pObjManager->pPlayer->getVehicle() != 0 ) {
    this->checkVehicleGameObjectCollisions();
    return;
  }
  
  // First we check if player collides to other objects
  Vec2D plaPos = m_pObjManager->pPlayer->position();
  int playerSector = Utils::sectorNumber( plaPos );
  GameObject* pP = m_pObjManager->pPlayer;
  
  // The player can collide to objects that are within the same sector as he
  // himself is.
  for ( int i=0; i < m_sectorVector.at(playerSector).size(); i++ ) {
    GameObject* pO = m_sectorVector.at(playerSector).at(i);
    if ( pO->state() == GameObject::STATE_KILLED ) {
      // Player can stand over a killed objects.
      continue;
    }
    
    // Check the object type and decide if player can stand over the object.
    switch ( pO->objectType() ) {
      case ( ObjectID::TYPE_SLIMEALIEN ):
      case ( ObjectID::TYPE_HIDDENWORMALIEN ):
      case ( ObjectID::TYPE_DECORATIVEOBJECT ):
      case ( ObjectID::TYPE_MEDIUMWORMALIEN ):
      case ( ObjectID::TYPE_SIMON ):
      case ( ObjectID::TYPE_ERIC ):
      case ( ObjectID::TYPE_YOUKO ):
      case ( ObjectID::TYPE_MINE ):
      case ( ObjectID::TYPE_FIGHTER ):
      case ( ObjectID::TYPE_WINGEDALIEN ):
      case ( ObjectID::TYPE_SMALLWORMALIEN ):
      case ( ObjectID::TYPE_CIVILIAN ): {
        // With these objects the player cannot collide.
        break;
      }
      case ( ObjectID::TYPE_GUARDIANTANK ):
      case ( ObjectID::TYPE_MACHINEGUN ):
      case ( ObjectID::TYPE_STARGATE ):
      case ( ObjectID::TYPE_SENTRYGUN ):
      case ( ObjectID::TYPE_CAR ): {
        // With these object the collision moves the player only.
        Vec2D diffVec = plaPos - pO->position();
        if ( fabs( diffVec.x() ) < 70 && fabs( diffVec.y() ) < 70 ) {
          if ( diffVec.length() < pO->boundingSphere() ) {
            // The objects collide.
            diffVec.norm();
            diffVec *= 4;
            pP->setForce( diffVec );
          }
        }
        break;
      }
      default: {
        Vec2D diffVec = plaPos - pO->position();
        if ( fabs( diffVec.x() ) < 70 && fabs( diffVec.y() ) < 70 ) {
          if ( diffVec.length() < pP->boundingSphere() ) {
            // The objects collide.
            diffVec.norm();
            pP->setForce( diffVec );
            pO->setForce( Vec2D() - diffVec );
          }
        }
        break;
      }
    }
  }
}


void War::checkVehicleGameObjectCollisions()
{
  Vec2D plaPos = m_pObjManager->pPlayer->position();
  int playerSector = Utils::sectorNumber( plaPos );
  Player* pP = m_pObjManager->pPlayer;
  GameObject* pV = pP->getVehicle();
  float bsphere = pV->boundingSphere();
  bool crushmode = false;
  
  if ( pV->objectType() == ObjectID::TYPE_TANK ) {
    // Collision with tank kills the normal objects only when the tank
    // is moving.
    BaseController* pC = pP->getController();
    if ( pC->forward() != 0 || pC->backward() != 0 ) {
      crushmode = true;
    }
  }
  
  for ( int i=0; i < m_sectorVector.at( playerSector ).size(); i++ ) {
    GameObject* pO = m_sectorVector.at( playerSector ).at( i );
    if ( pO->state() == GameObject::STATE_KILLED ) {
      continue;
    }
    Vec2D diffVec( plaPos );
    diffVec -= pO->position();
    if ( fabs(diffVec.vx) < bsphere && fabs(diffVec.vy) < bsphere ) {
      switch ( pO->objectType() ) {
        case ( ObjectID::TYPE_PLAYER ):
        case ( ObjectID::TYPE_FIGHTER ):
        case ( ObjectID::TYPE_WINGEDALIEN ): {
          // With these objects we cannot collide
          break;
        }
        case ( ObjectID::TYPE_STARGATE ):
        case ( ObjectID::TYPE_TANK ):
        case ( ObjectID::TYPE_SENTRYGUN ): {
          // These object-types cannot be crushed.
          break;
        }
        default: {
          if ( crushmode == true ) {
            pO->kill();
          } else {
            diffVec.norm();
            diffVec *= 4;
            pO->setForce( Vec2D() - diffVec );
          }
          break;
        }
      }
    }
  }
}


void War::checkBulletCollisions()
{
  // The bullets can collide only to the objects that are within the same
  // sector as the bullet itself.
  BulletTable::BulletIter it = WarGlobals::pBulletManager->bulletBegin();
  
  while ( it != WarGlobals::pBulletManager->bulletEnd() ) {
    Bullet* pB = (*it);
    int secNum = Utils::sectorNumber( pB->iPosition );
    bool delBullet = false;
    for ( int i=0; i < m_sectorVector.at(secNum).size(); i++ ) {
      GameObject* pO = m_sectorVector.at(secNum).at( i );
      if ( this->checkBulletGameObjectCollision( pB, pO ) == true ) {
        // We should delete the bullet.
        delBullet = true;
        break;
      }
    }
    
    if ( delBullet == true ) {
      it = WarGlobals::pBulletManager->deleteBullet( it );
      
    } else {
      it++;
      
    }
  }
}



/** Checks the collision between given bullet and gameobject.
 */
bool War::checkBulletGameObjectCollision( Bullet* pB, GameObject* pO )
{
  Vec2D diffVec( pO->position() );
  diffVec -= pB->iPosition;
  
  if ( diffVec.length() < pO->boundingSphere() + pB->iRadius ) {
    return this->handleBulletHit( pB, pO );
  }
  return false;
}


bool War::handleBulletHit( Bullet* pB, GameObject* pO )
{
  bool deleteBullet = false;
  
  // The grenade fragments give a little boost to the objects
  if ( pB->iType == Bullet::EGrenade &&
       pO->hasProperty( GameObject::PROP_GRENADEFORCE ) ) {
     pO->setForce( pB->velocity() );
  }
  // If the object is killed, or dying, we return.
  if ( pO->state() == GameObject::STATE_KILLED ||
       pO->state() == GameObject::STATE_DYING ) {
    return false;
  }
  
  // Inform the GameObject about the bullethit.
  bool h = pO->hitByBullet( pB );
  if ( h == true ) {
    // If the Cannonball shot by a Tank hits someone, it explodes.
    if ( pB->iType == Bullet::ETankBullet ) {
      pB->iDone = true;
    }
    
    // The bullet hit occured. Now we check the bullet type and tell the
    // collision manager to delete the bullet or leave it.
    switch ( pB->iType ) {
      case ( Bullet::ECrowbar ): {
        // The bullet is from crowbar. Make the hitting sound.
        Sound::playSample( SMP_HAMMERHIT, false );
        deleteBullet = true;
        break;
      }
      case ( Bullet::EFlameThrower ): {
        // The flamethower's "bullets" slow down as they hit someone. We don't
        // delete it.
        Vec2D spd( pB->velocity() );
        spd.scale( 0.4 );
        pB->setVelocity( spd );
        deleteBullet = false;
        break;
      }
      default: {
        deleteBullet = true;
        break;
      }
    }
  }
  
  
  // Manage the statistics
  if ( pB->iOwner != 0  && h == true) {
    if ( pB->iOwner->objectType() == ObjectID::TYPE_PLAYER ) {
      // The shooter was player.
      // If player used crowbar, we ignore the hit.
      if ( !( pB->iType == Bullet::ECrowbar ||
              pB->iType == Bullet::EFlameThrower ||
              pB->iType == Bullet::EGrenade ) ) {
        WarGlobals::gameStats.updateStatistics( Statistics::HITS, 0 );
      }
      
      // If the target did die, we grant player the killing bonuses and more
      // bullet time
      if ( pO->health() < 0 ) {
        WarGlobals::gameStats.updateStatistics( Statistics::KILLS,
                                                pO->objectType() );
        WarGlobals::increaseBulletTime( 40 );
        m_comboMeter->advance( pO->objectType() );
        int scoresAnimUid = ScoresAnimation::getKillScoresAnimationUid( pO->objectType() );
        if ( scoresAnimUid != -1 ) {
          const Animation& anim = GameAnims::findAnimation( AnimId::KScoresAnimations, scoresAnimUid );
          AnimPlayer::spawn( anim, pO->position(), Vec2D(0,-0.5), 0 );
        }
      }
    }
  }
  
  // If the health of the target object has reached 0, we kill it.
  if ( pO->health() < 0 ) {
    pO->kill();
  }
  
  return deleteBullet;
}


void War::checkPlayerLightfenceCollisions()
{
  Vec2D plaPos = m_pObjManager->pPlayer->position();
  LightFence* pF = WarGlobals::pFenceManager->lightfenceTouch( plaPos, 10 );
  if ( pF != 0 ) {
    // Direction from the fence.
    Vec2D tmpVec( pF->middlePos() );
    tmpVec -= plaPos;
    tmpVec.norm();
    
    int damage = 0;
    switch ( pF->fenceType() ) {
      case ( ElectricFence::FENCETYPE_ELECTRIC ): {
        tmpVec *= -ElectricFence::s_shockMagnitude;
        damage = ElectricFence::s_shockDamage;
        break;
      }
    }
    m_pObjManager->pPlayer->setForce( tmpVec );
    m_pObjManager->pPlayer->move( tmpVec );
    ParticleBlood* pPB = new ParticleBlood( plaPos, tmpVec, 8, Color(140,80,30) );
    WarGlobals::pPartManager->addSystem( pPB );
    
    // And finally we cause some damage.
    m_pObjManager->pPlayer->causeDamage( damage );
  }
}


void War::updateHud()
{
  GameObject* pV = m_pObjManager->pPlayer->getVehicle();
  if ( pV == 0 ) {
    m_pHud->healthLevel( m_pObjManager->pPlayer->health() );
  } else {
    m_pHud->healthLevel( pV->health() );
  }
  
  float shield = 0;
  float maxShield = 0;
  m_pObjManager->pPlayer->getShieldValues( shield, maxShield );
  m_pHud->setBodyArmor( static_cast<int>(shield), static_cast<int>(maxShield) );
  
  m_pHud->activeWeapon( m_pObjManager->pPlayer->getWeapon() );
  m_pHud->ammocount( m_pObjManager->pPlayer->getAmmunition() );
  m_pHud->grenadecount( m_pObjManager->pPlayer->getGrenadeCount() );
  m_pHud->setCrosshairSprite( GfxManager::findRleSprite(GfxId::KHudSymbols,
                                                        HudSymbol_Crosshair1) );
  
  Objective* pO = m_pLevel->getObjective( m_currentObjective );
  if ( pO != 0 ) {
    int t = pO->getTimer();
    if ( t > 0 ) {
      m_pHud->countdownValue( t );
    } else {
      m_pHud->countdownValue( -1 );
    }
  }
  
  m_pHud->objectiveMessage( pO->getMessage() );
  m_pHud->bulletTime( WarGlobals::bulletTimeLeft );
  m_pHud->update();
  
  // We scroll the map to the right position.
  Vec2D plaPos = m_pObjManager->pPlayer->position();
  int fromX = static_cast<int>( plaPos.x() );
  int fromY = static_cast<int>( plaPos.y() );

  fromX -= Display::scrWidth() / 2;
  fromY -= Display::scrHeight() / 2;

  if (fromX < 0) fromX = 0;
  if (fromY < 0) fromY = 0;
  if (fromX + Display::scrWidth() >= Map::getWidth(Map::IN_PIXELS)) {
    fromX = Map::getWidth(Map::IN_PIXELS) - Display::scrWidth() -1;
  }
  if (fromY + Display::scrHeight() >= Map::getHeight(Map::IN_PIXELS)) {
    fromY = Map::getHeight(Map::IN_PIXELS) - Display::scrHeight() - 1;
  }

  // Store the scrolling parameters for later use
  if ( WarGlobals::screenLockMode == true ) {
    Map::scrollX = WarGlobals::screenLockPos.intX();
    Map::scrollY = WarGlobals::screenLockPos.intY();
    
    // Make sure the player is inside the screen
    Rect2D rect;
    rect.topleft = WarGlobals::screenLockPos;
    rect.topleft += Vec2D( 32,32 );
    rect.bottomright = WarGlobals::screenLockPos;
    rect.bottomright += Vec2D( 608,448 );
    if ( rect.pointInside( plaPos ) == false ) {
      // Move the player inside the rectangle
      if ( plaPos.vx < rect.topleft.vx ) {
        plaPos.vx = rect.topleft.vx;
      }
      if ( plaPos.vx > rect.bottomright.vx ) {
        plaPos.vx = rect.bottomright.vx;
      }
      if ( plaPos.vy < rect.topleft.vy ) {
        plaPos.vy = rect.topleft.vy;
      }
      if ( plaPos.vy > rect.bottomright.vy ) {
        plaPos.vy = rect.bottomright.vy;
      }
      m_pObjManager->pPlayer->position( plaPos );
    }
    
  } else {
    // If player has captured a vehicle or if he is in sniper mode,
    // he can adjust the visible range a bit
    GameObject* vehicle = m_pObjManager->pPlayer->getVehicle();
    if ( vehicle != 0 || m_pObjManager->pPlayer->sniperMode() == true ) {
      // Player has a vehicle or is in sniper mode. Normally the Player-class
      // is responssible of setting the crosshair position but when operating
      // in sniper mode or when player has captured a vehicle, we override
      // that.
      Vec2D bottomRight( Map::getWidth(Map::IN_PIXELS) - Display::scrWidth(),
                         Map::getHeight(Map::IN_PIXELS) - Display::scrHeight() );
      Rect2D mapRect( bottomRight );
      
      plaPos = m_pObjManager->pPlayer->position();
      Vec2D aimPos( m_pHud->m_cursorpos );
      aimPos -= Vec2D( Display::scrWidth()/2, Display::scrHeight()/2 );
      if ( m_pObjManager->pPlayer->sniperMode() == true ) {
        // In sniper mode the range is wider.
        aimPos *= Consts::SNIPERMODE_RANGE_FACTOR;
      }
      aimPos += plaPos;
      plaPos += aimPos;
      plaPos *= 0.5;
      plaPos -= Vec2D( Display::scrWidth()/2, Display::scrHeight()/2 );
      mapRect.snapInside( plaPos );
      Map::scrollX = plaPos.intX();
      Map::scrollY = plaPos.intY();
      
    } else if ( WarGlobals::freeScrollMode == false ) {
      Map::scrollX = fromX;
      Map::scrollY = fromY;
    }
  }
  
  // If player is in sniper-mode, we use different crosshair sprite.
  if ( m_pObjManager->pPlayer->sniperMode() == true ) {
    m_pHud->setCrosshairSprite(
        GfxManager::findRleSprite(GfxId::KHudSymbols, HudSymbol_Crosshair2) );
  }
}


void War::redraw()
{
  switch ( m_menupage ) {
    case ( PAGE_GAME ): {
      this->redrawGame();
      break;
    }
    case ( PAGE_PAUSEMENU ): {
      this->redrawPauseMenu();
      break;
    }
    case ( PAGE_PLAYERDIED ): {
      this->redrawPauseMenu();
      break;
    }
    case ( PAGE_LEVELCOMPLETE ): {
      this->redrawPauseMenu();
      break;
    }
    case ( PAGE_LEVELFADEIN ): {
      this->redrawLevelFadeIn();
      break;
    }
    case ( PAGE_GAMEOVERSCREEN ): {
      this->redrawGameOverScreen();
      break;
    }
  }
}

void War::redrawGame()
{
  m_pRedrawQueue->clear();
  
  this->drawMap( 0 );
  this->drawMap( 1 );
  
  // Add objects to the redraw queue
  m_pObjManager->redrawObjects( m_pRedrawQueue );
  
  // Start drawing the objects. First we draw killed objects
  m_pRedrawQueue->redraw( RedrawQueue::PRI_KILLED, Display::buffer );
  
  // Draw the bonus items
  this->drawBonuses();
  
  // Draw objects that have priority below normal
  m_pRedrawQueue->redraw( RedrawQueue::PRI_BELOW_NORMAL, Display::buffer );
  
  // Draw most of the gameobjects
  m_pRedrawQueue->redraw( RedrawQueue::PRI_NORMAL, Display::buffer );
  
  // Draw player
  m_pRedrawQueue->redraw( RedrawQueue::PRI_ABOVE_NORMAL, Display::buffer );
  
  // Draw the bullets and grenades
  WarGlobals::pBulletManager->redraw( Display::buffer );
  
  // Draw the extra animations
  AnimPlayer::redraw( Display::buffer );
  
  // Draw the particles.
  WarGlobals::pPartManager->redraw( Display::buffer,
                                    Vec2D(Map::scrollX, Map::scrollY) );  
  // Draw objects that are flying
  m_pRedrawQueue->redraw( RedrawQueue::PRI_FLYING, Display::buffer );

  // Draw the static lights
  if ( m_lightTable != 0 ) {
    m_lightTable->redraw( Display::buffer, Vec2D(Map::scrollX, Map::scrollY) );
  }
  
  // Draw the dynamic shadows
  if ( m_pLevel->nightModeOn == false ) {
    // Don't draw shadows when night mode is on.
    Vec2D center( Map::scrollX + 320, Map::scrollY + 240 );
    WarGlobals::pShadowManager->applyShadows( Display::buffer, center );
  }
  // And finally we draw the topmost decorative map layer.
  this->drawMap( 2 );
  
  // And finally objects that are flying high
  m_pRedrawQueue->redraw( RedrawQueue::PRI_FLYING_HIGH, Display::buffer );

  Vec2D scroll( Map::scrollX, Map::scrollY );
  WarGlobals::pFenceManager->redraw( Display::buffer, scroll );
  
  // Draw the weather effects
  if ( m_pWeather != 0 ) {
    m_pWeather->redraw();
  }
  
  if ( m_pLevel->nightModeOn == true ) {
    LightBeam* beam = m_pObjManager->pPlayer->FlashLightBeam();
    if ( beam != 0 ) {
      Vec2D playerPos = m_pObjManager->pPlayer->position();
      playerPos -= Vec2D( Map::scrollX, Map::scrollY );
      int beamWidth = 0;
      int beamHeight = 0;
      beam->lightBeamSize( beamWidth, beamHeight );
      int x1 = playerPos.intX() - (beamWidth/2);
      int y1 = playerPos.intY() - (beamHeight/2);
      int x2 = playerPos.intX() + (beamWidth/2);
      int y2 = playerPos.intY() + (beamHeight/2);
      
      rectfill( Display::buffer, 0,0, 640,y1, 0 );
      rectfill( Display::buffer, 0,y1, x1,480, 0 );
      rectfill( Display::buffer, x2,y1, 640,480, 0 );
      rectfill( Display::buffer, x1, y2, x2,480, 0 );
      
      beam->applyLightBeam( Display::buffer, playerPos );
    }
  }
  // Draw heads up display
  m_pHud->redraw( Display::buffer );
  m_comboMeter->redraw( Display::buffer );
  
  // If we're in free scroll mode, we draw a label at the bottom of the
  // screen.
  if ( WarGlobals::freeScrollMode == true ) {
    textout_centre_ex( Display::buffer, font, "- FREE SCROLL -", 320,470,
                       makecol(255,255,255),-1 );
  }
}


void War::redrawPauseMenu()
{
  if ( m_pPauseBitmap != 0 ) {
    blit( m_pPauseBitmap, Display::buffer, 0,0, 0,0, 640,480 );
  }

  if ( m_menupage == PAGE_PLAYERDIED && m_pDeadTitle != 0 ) {
    blit( m_pDeadTitle, Display::buffer, 0,0, 240,25, 400,87 );
    
  } else if ( m_menupage == PAGE_LEVELCOMPLETE ) {
    blit( m_pLevelCompleteTitle, Display::buffer, 0,0, 189,25, 451,87 );
    
  }

  Statistics& rS = WarGlobals::gameStats;
  BMFont* titleFont = GfxManager::titleFont;
  
  // Print the statistics
  TPrinter out( Display::buffer );
  out.font( titleFont, "normal30" );
  out.align( TPrinter::ALIGN_CENTER );
  out.pos( 210,130 );
  out.print( "total score" );

  out.pos( 210,230 );
  out.print( "level statistics" );

  out.pos( 210,330 );
  out.print( "overall statistics" );

  // Print total score
  out.font( titleFont, "dark30" );
  out.pos( 210,170 );
  out.print( rS.score() );

  // print the level statistics titles
  out.font( titleFont, "menuitems" );
  out.align( TPrinter::ALIGN_RIGHT );
  out.pos( 190,260 );
  out.print( "shots" );
  
  out.pos( 190,280 );
  out.print( "hits" );
  
  out.pos( 190,300 );
  out.print( "kills" );
  
  out.pos( 190,360 );
  out.print( "shots" );

  out.pos( 190,380 );
  out.print( "hits" );
  
  out.pos( 190,400 );
  out.print( "kills" );
  
  // level statistics values
  out.font( titleFont, "menuitems_sel" );
  out.align( TPrinter::ALIGN_LEFT );
  out.pos( 230, 260 );
  out.print( rS.getLevelStats( Statistics::SHOTS ) );
  
  out.pos( 230, 280 );
  out.print( rS.getLevelStats( Statistics::HITS ) );
  
  out.pos( 230, 300 );
  out.print( rS.getLevelStats( Statistics::KILLS ) );
  
  out.pos( 230,360 );
  out.print( rS.getOverallStats( Statistics::SHOTS ) );
  
  out.pos( 230,380 );
  out.print( rS.getOverallStats( Statistics::HITS ) );
  
  out.pos( 230,400 );
  out.print( rS.getOverallStats( Statistics::KILLS ) );
  
  // Print the time
  int levsecs = rS.levelTimer() / 40;
  int levmins = levsecs / 60;
  levsecs = levsecs % 60;
  
  string secstr( "00" );
  string minstr( "00" );
  secstr[0] = static_cast<char>( levsecs / 10 ) + '0';
  secstr[1] = static_cast<char>( levsecs % 10 ) + '0';
  minstr[0] = static_cast<char>( levmins / 10 ) + '0';
  minstr[1] = static_cast<char>( levmins % 10 ) + '0';  
  
  out.font( titleFont, "normal30" );
  out.align( TPrinter::ALIGN_RIGHT );
  out.pos( 240,440 );
  out.print( "level time" );
  out.font( titleFont,  "dark30" );
  out.pos( 260,440 );
  out.align( TPrinter::ALIGN_LEFT );
  out.print( minstr );
  out.print( ":" );
  out.print( secstr );
  
  // Print number of lives
  out.align( TPrinter::ALIGN_LEFT );
  out.font( titleFont, "menuitems" );
  out.pos( 450,120 );
  out.print( "lives left " );
  out.font( titleFont, "menuitems_sel" );
  out.print( WarGlobals::numberOfLives );
  
  // Print the menuoptions to the left edge.
  if ( m_menupage != PAGE_LEVELCOMPLETE ) {
    out.align( TPrinter::ALIGN_CENTER );
    if ( m_selectedMenuItem == 0 ) {
      out.font( titleFont, "menuitems_sel" );
    } else {
      out.font( titleFont, "menuitems" );
    }
    out.pos( 530,220 );
    if ( m_menupage == PAGE_PAUSEMENU ) {
      out.print( "resume game" );
    
    } else if ( m_menupage == PAGE_PLAYERDIED ) {
      out.print( "resurrect" );
    
    }
  
    if ( m_selectedMenuItem == 1 ) {
      out.font( titleFont, "menuitems_sel" );
    } else {
      out.font( titleFont, "menuitems" );
    }
    out.pos( 530,260 );
    out.print( "settings" );
  
    if ( m_selectedMenuItem == 2 ) {
      out.font( titleFont, "menuitems_sel" );
    } else {
      out.font( titleFont, "menuitems" );
    }
    out.pos( 530,300 );
    out.print( "quit" );
    
  } else {
    out.pos( 510,230 );
    out.align( TPrinter::ALIGN_CENTER );
    out.font( titleFont, "menuitems" );
    out.print( "awarded" );
    out.pos( 510,250 );
    out.print ( "timebonus" );
    
    out.font( titleFont, "dark30" );
    out.pos( 512,270 );
    if ( WarGlobals::timeBonuses < 0 ) {
      out.print( "0" );
    } else {
      out.print( WarGlobals::timeBonuses );
    }
  }
}


void War::redrawLevelFadeIn()
{
  this->redrawGame();
  int upperY = m_currentScanLine;
  int lowerY = 240 + (240 - m_currentScanLine);

  for ( int y=0; y < upperY; y++ ) {
    blit( Display::buffer, Display::buffer, 0,upperY, 0,y, 640,1 );
  }
  for ( int y=480; y > lowerY; y-- ) {
    blit( Display::buffer, Display::buffer, 0,lowerY, 0,y, 640,1 );
  } 
}


void War::redrawGameOverScreen()
{
  this->redrawGame();
  RLE_SPRITE* gameOver = GfxManager::findRleSprite(GfxId::KGameOverLabel);
  int x = 320 - (gameOver->w/2);
  int y = 200 - (gameOver->h/2);
  
  int alpha = 2 * m_gameOverCounter;
  if ( alpha > 255 ) {
    draw_rle_sprite( Display::buffer, gameOver, x,y );
  } else {
    set_trans_blender( 0,0,0, alpha );
    draw_trans_rle_sprite( Display::buffer, gameOver, x,y );
  }
}


void War::drawMap( int l )
{ 
  if ( l == 0 ) {
    Map::redraw( Map::scrollX, Map::scrollY, 0,0,
                  Display::scrWidth(),Display::scrHeight(), false );
    
  } else if ( l == 1 ) {
    Map::redrawFG( Map::scrollX, Map::scrollY, 0,0,
                   Display::scrWidth(),Display::scrHeight() );
    
  } else if ( l == 2 ) {
    if ( Map::selectLayer( Map::EDecorative ) != -1 ) {
      Map::redraw( Map::scrollX,Map::scrollY, 0,0,
                   Display::scrWidth(),Display::scrHeight(), true );
    }
    Map::selectLayer( Map::EBackground );
  }
}


void War::drawBonuses()
{
  for ( int i=0; i < WarGlobals::bonusList.size(); i++ ) {
    if ( WarGlobals::bonusList.at(i) != 0 ) {
      WarGlobals::bonusList.at(i)->redraw( Display::buffer );
    }
  }
}


void War::showLevelStory()
{
  StoryViewer* storyViewer = new StoryViewer();
  string bgfile = m_pLevel->titlepicfile;
  string storyfile = m_pLevel->storypicfile;  
  storyViewer->showStory( bgfile, storyfile );
  delete storyViewer;
}

void War::showCredits()
{
  while ( key[KEY_ESC] || key[KEY_SPACE] || key[KEY_ENTER] ) {
    rest( 50 );
  }
  CreditsViewer* viewer = new CreditsViewer();
  viewer->showCredits();
  delete viewer;
}

void War::showDebugInfo()
{
  clear_bitmap( screen );
  
  textprintf_ex( screen, font, 10,10, makecol(255,255,255),0,
                 "Number of bullets: %d", WarGlobals::pBulletManager->bulletCount() );
                 
  textprintf_ex( screen, font, 10,20, makecol(255,255,255),0,
                 "Total objectcount: %d", GameObject::s_numberOfGameObjects );
                 
  while ( !key[KEY_SPACE] ) {
    MP3Player::poll();
  }
}


void War::whileKey( int scan )
{
  while ( key[scan] ) {
    if ( Settings::musicOn ) {
      MP3Player::poll();
    }
  }
}


void War::showKeyInfo()
{
  if ( m_showKeyInfo == false ) {
    return;
  }
  
  m_pHud->addMessage( string( "RIGHT MB -  EXPLODE GRENADES" ), 276 );
  m_pHud->addMessage( string( "LEFT MB  -  SHOOT" ), 272 );
  m_pHud->addMessage( string( "1...9    -  CHANGE WEAPON" ), 268 );
  m_pHud->addMessage( string( "MOUSE    -  TURN AND AIM" ), 264 );
  m_pHud->addMessage( string( "F        -  THROW GRENADE" ), 260 );
  m_pHud->addMessage( string( "A,S,D,W  -  MOVE" ), 256 );
}


void War::findVehicleToCapture()
{
  // Find if there's a tank near us and capture it
  for ( int i=0; i < m_pObjManager->objectList.size(); i++ ) {
    GameObject* obj = m_pObjManager->objectList.at(i);
    bool capturableVehicle = false;
    
    // Check the type of this vehicle and decide whether it could be
    // captured or not.
    switch ( obj->objectType() ) {
      case ( ObjectID::TYPE_MACHINEGUN ):
      case ( ObjectID::TYPE_PLAYERCAR ):
      case ( ObjectID::TYPE_TANK ): {
        capturableVehicle = true;
        break;
      }
      default: {
        capturableVehicle = false;
        break;
      }
    }
    
    if ( capturableVehicle == true ) {
      // This object is capturable vehicle. If it's close enough,
      // we capture it.
      Vec2D distVec( m_pObjManager->pPlayer->position() );
      distVec -= obj->position();
      if ( distVec.length() < 50 ) {
        // This vehicle is close enough so that we can capture it.
        MCapturable* vehicle = dynamic_cast<MCapturable*>( obj );
        if ( vehicle != 0 ) {
          if ( vehicle->canBeCaptured() == true ) {
            // Player captures this vehicle.
            vehicle->capture( m_pObjManager->pPlayer );
            return;
          }
        }
      }
    }
  }
}


void War::grantCheatBonuses()
{
  bool soundStatus = Sound::soundsOn;
  Sound::soundsOn = false;
  vector<BonusObject*> bonuses;
  bonuses.push_back( new BonusObject( BonusObject::ROCKETLAUNCHER ) );
  bonuses.push_back( new BonusObject( BonusObject::SHOTGUN ) );
  bonuses.push_back( new BonusObject( BonusObject::FLAMETHROWER ) );
  bonuses.push_back( new BonusObject( BonusObject::MINIGUN ) );
  bonuses.push_back( new BonusObject( BonusObject::UZI ) );
  bonuses.push_back( new BonusObject( BonusObject::GRENADE ) );
  bonuses.push_back( new BonusObject( BonusObject::SNIPERRIFLE ) );
  bonuses.push_back( new BonusObject( BonusObject::RIFLE ) );
  for ( int i=0; i < bonuses.size(); i++ ) {
    for ( int j=0; j < 5; j++ ) {
      m_pObjManager->pPlayer->pickupBonus( bonuses.at(i) );
    }
    delete bonuses.at(i);
    bonuses.at(i) = 0;
  }
  Sound::soundsOn = soundStatus;
}


void War::createLightModels()
{
  if ( m_lightTable == 0 ) {
    return;
  }
  LOG_MESSAGE( "    Registering lightsource models..." );
  try {
    LightSourceModel* pointModel = new PointLightModel( LightModelUid::KStreetLight,
                                                        96,
                                                        Color(255,255,220) );
    m_lightTable->registerLightSourceModel( pointModel );  
    
  } catch ( ... ) {
    LOG_MESSAGE( "    Lightsource creation FAILED!" );
    
  }
  
  LOG_MESSAGE( "    Lightsources registered." );
}

void War::setMp3Volume( int aVolume )
{
  MP3Player::volume( aVolume );
}

void War::playMp3File( int aSongNumber )
{
  if ( aSongNumber >= m_playlist.size() && aSongNumber != -1 ) {
    return;
  }
  
  MP3Player::stop();
  MP3Player::setMP3( 0 );
  if ( aSongNumber < 0 ) {
    return;
  }
  m_playlist.at(aSongNumber)->open();
  MP3Player::setMP3( m_playlist.at( aSongNumber ) );
  MP3Player::play();
}

int War::playbackVolume() const
{
  return MP3Player::volume();
}

}   // end of namespace

/**
 * Version history
 * ===============
 * $Log: war.cpp,v $
 * Revision 1.17  2006/08/13 21:03:40  lamminsa
 * Game Over screen added.
 *
 * Revision 1.16  2006/08/13 10:33:47  lamminsa
 * In main loop the rest(10) was replaced by rest(0). Fixes the crappy
 * update rate on Geforce 6600 GT
 *
 * Revision 1.15  2006/07/27 20:30:36  lamminsa
 * MP3 playback volume adjustments.
 *
 * Revision 1.14  2006/06/17 21:47:49  lamminsa
 * Fixed the frameskipping bug and implemented the flashlight
 * feature.
 *
 * Revision 1.13  2006/05/30 14:56:48  lamminsa
 * Credits are shown when the last level has been played through.
 *
 * Revision 1.12  2006/05/15 19:22:25  lamminsa
 * Implements the MMp3PlayerProvider interface.
 *
 * Revision 1.11  2006/05/08 21:30:01  lamminsa
 * SlimeAlien added.
 *
 * Revision 1.10  2006/04/25 17:55:59  lamminsa
 * HiddenWormAlien collision detection added. Rocketlauncher as cheat
 * bonus added.
 *
 * Revision 1.9  2006/04/12 21:12:46  lamminsa
 * updateHud() method updates the player's bodyarmor values.
 *
 * Revision 1.8  2006/04/03 21:01:24  lamminsa
 * Scores and combo animations added.
 *
 * Revision 1.7  2006/03/29 22:30:58  lamminsa
 * Code cleanup and GuardianTank added.
 *
 * Revision 1.6  2006/03/16 21:31:42  lamminsa
 * Added a rest() call to the mainloop that prevents WeWantWar consuming
 * all the available processor time.
 *
 * Revision 1.5  2006/02/26 07:38:39  lamminsa
 * Level fade in animation.
 *
 * Revision 1.4  2006/02/17 23:08:39  lamminsa
 * Stargate added.
 *
 * Revision 1.3  2006/02/11 23:42:07  lamminsa
 * New lighttable implementation.
 *
 * Revision 1.2  2006/02/08 21:42:57  lamminsa
 * Implemented totally new graphic and animation system.
 *
 * Revision 1.1.1.1  2006/01/21 23:02:43  lamminsa
 * no message
 *
 * Revision 1.3  2006-01-10 22:58:18+02  lamminsa
 * Player collision with barrels modified.
 *
 * Revision 1.2  2005-12-30 23:44:53+02  lamminsa
 * <>
 *
 * Revision 1.1  2005-11-13 14:52:33+02  lamminsa
 * Bullet deletion was changed.
 *
 */
